using System;
using System.Collections.Generic;
using System.Text;

namespace Intemi.InTrees.Implements
{
    public class Pruning2Config : IPruningConfiguration
    {
        private int nodeCount;
        public int NodeCount
        {
            get { return nodeCount; }
            set { nodeCount = value; }
        }
    }

    //public class Pruning2Result : IPruningResult
    //{
    //    private double accuracy;
    //    public double Accuracy
    //    {
    //        get { return accuracy; }
    //        set { accuracy = value; }
    //    }
    //}

    public class Pruning2Table : IPruningTable
    {
        public Pruning2Table(int maxNodeCount)
        {
            this.maxNodeCount = maxNodeCount;
            accuracy = new double[maxNodeCount+1];
        }

        //private IPruningConfiguration[] configuration;
        //public IPruningConfiguration[] Configuration
        //{
        //    get { return configuration; }
        //}

        private double[] accuracy;
        public double[] Accuracy
        {
            get { return accuracy; }
        }

        private int maxNodeCount;
        public int MaxNodeCount
        {
            get { return maxNodeCount; }
            //set { maxNodeCount = value; }
        }
    }

    public class Pruning2 : IPruning
    {
        /// <summary>
        /// Przycina drzewo stosujc wedug kryteriw zawartych w konfiguracji przycinania
        /// </summary>
        /// <param name="tree">Drzewo do przycicia</param>
        /// <param name="config">Konfiguracja przycinania</param>
        /// <param name="shouldTerminate">Polecenie przerwania procedury</param>
        /// <returns>Przycite drzewo</returns>
        public DecisionTree Prune(DecisionTree tree, IPruningConfiguration config, IDataTable trainDT, ref bool shouldTerminate)
        {
            List<Node<DTNode>> candidates = new List<Node<DTNode>>();
            //IEnumerator<Node<DTNode>> treeEn = tree.GetBreadthFirstEnumerator();

            int nodeCount = tree.CountNodes(false);
            candidates = FindCandidates(tree.Root, ref shouldTerminate); 

            // W tym momencie mamy 
            // 1. Ilo wzw w drzewie nie bdcych limi
            // 2. Kolekcj wzw kandydujcych do przycicia

            Node<DTNode> bestNode;
            double bestAcc;

            int maxCount = (config as Pruning2Config).NodeCount;
            while (nodeCount > maxCount)
            {
                // szukamy najlepszego wza do przycicia
                bestNode = null;
                bestAcc = 0;
                foreach (Node<DTNode> node in candidates)
                {
                    //node.Data.DataRecalculate(trainDT, ref shouldTerminate);
                    if (node.Data.Accuracy > bestAcc)
                    {
                        bestNode = node;
                        bestAcc = node.Data.Accuracy;
                    }
                }

                // przycinamy wze
                if (bestNode != null)
                {
                    bestNode.Data.IsLeaf = true;
                    bestNode.Children.Clear();
                    
                    // wze sta si liciem - usuwamy go z listy kandydatw
                    candidates.Remove(bestNode);

                    // ilo wzw w drzewie zmiejsza si
                    nodeCount--;

                    // jeeli w wyniku przycicia wza jego rodzic sta si wzem posiadajcym tylko dzieci-licie
                    // rodzic staje si kandydatem do przycicia,
                    // ale jeeli nodeCount <= maxCount to nie ma sensu sprawdza, bo koczymy przycinanie
                    if ((nodeCount > maxCount) && IsCandidate(bestNode.Parent))
                    {
                        candidates.Add(bestNode.Parent);
                    }
                }
            }

            return (tree);
        }

        private List<Node<DTNode>> FindCandidates(Node<DTNode> node, ref bool shouldTerminate)
        {
            List<Node<DTNode>> candidates = new List<Node<DTNode>>();
            bool isCandidate = true;
            foreach (Node<DTNode> child in node.Children)
            {
                if (!child.Data.IsLeaf)
                {
                    isCandidate = false;
                    candidates.AddRange(FindCandidates(child, ref shouldTerminate));
                }
            }

            if (isCandidate)
                candidates.Add(node);

            return (candidates);
        }

        private bool IsCandidate(Node<DTNode> node)
        {
            bool isCandidate = true;
            foreach (Node<DTNode> node0 in node.Children)
            {
                if (!node0.Data.IsLeaf)
                {
                    isCandidate = false;
                    break;
                }
            }
            return (isCandidate);
        }

        /// <summary>
        /// Wskazuje optymaln konfiguracj przycinania
        /// </summary>
        /// <param name="tables">Tablice zawierajce konfiguracj przycinania i osignite przez nie wyniki klasyfikacji</param>
        /// <param name="shouldTerminate">Polecenie przerwania procedury</param>
        /// <returns>Optymalna konfiguracja</returns>
        public IPruningConfiguration SelectOptimalConfiguration(List<IPruningTable> tables, ref bool shouldTerminate)
        {
            // ustalamy najwiksz liczb wezw w drzewie
            int globalMaxNC = 0;
            int nodeCount;
            foreach (IPruningTable table in tables)
            {
                nodeCount = (table as Pruning2Table).MaxNodeCount;
                if (nodeCount > globalMaxNC)
                    globalMaxNC = nodeCount;
            }

            double[,] acc = new double[tables.Count,globalMaxNC+1];

            //double accur;

            // tworzymy dla kadej PruningTable tworzymy tabelk zawierajc osignit Acc dla danej iloci wzw
            Pruning2Table pTable;
            int i = 0;
            foreach (IPruningTable table in tables)
            {
                pTable = table as Pruning2Table; 
                for (int a = 0; a <= pTable.MaxNodeCount; a++)
                {
                    acc[i, a] = pTable.Accuracy[a];
                }
                i++;
            }

            // dla kadej iloci wzw obliczamy redni accuracy i szukamy najwyszej
            double[] avAcc = new double[globalMaxNC+1];
            //double maxAcc = 0;
            nodeCount=0;
            for (int k = 0; k <= globalMaxNC; k++)
            {
                for(int l=0;l<tables.Count;l++)
                {
                    if (acc[l,k]>0)
                        avAcc[k]+=acc[l,k];
                    else
                    {
                        // oznacza to e taka ilo wzw nie wystpia w ktrym zestawie, wic nie moe by ona optymaln
                        avAcc[k]=0;
                        break;
                    }
                }
                avAcc[k] = avAcc[k]/tables.Count;
                if (avAcc[k] > avAcc[nodeCount])
                {
                    nodeCount = k;
                }
            }

            Pruning2Config conf = new Pruning2Config();
            conf.NodeCount = nodeCount;

            return (conf);
        }

        /// <summary>
        /// Przycina drzewo w rnych konfiguracjach i zwraca tablic zawierajc tego efekty
        /// </summary>
        /// <param name="tree">Drzewo</param>
        /// <param name="shouldTerminate">Polecenie przerwania procedury</param>
        /// <returns>Tablica zawierajca kongiguracj przycinania i osignite przez nie wyniki klasyfikacji</returns>
        public IPruningTable BuildPruningTable(DecisionTree tree, IDataTable trainDT, IDataTable testDT, ref bool shouldTerminate)
        {
            int nodeCount = tree.CountNodes(false);
            Pruning2Table pTable = new Pruning2Table(nodeCount);
            Pruning2Config config = new Pruning2Config();

            pTable.Accuracy[nodeCount]=tree.Accuracy(testDT,ref shouldTerminate);
            while (nodeCount > 0)
            {
                config.NodeCount = nodeCount - 1;
                tree = Prune(tree, config, trainDT, ref shouldTerminate);
                nodeCount--;
                pTable.Accuracy[nodeCount] = tree.Accuracy(testDT, ref shouldTerminate);
            }

            return (pTable);
        }
    }
}
